#ifndef __CMenuWindow__
#define __CMenuWindow__

#include "CWindow.hpp"
#include "../Controls/CSubMenuMenuItem.hpp"
#include "../Controls/SMenuColours.hpp"
#include "../Listeners/IMenuListener.hpp"

//	===========================================================================

using Exponent::GUI::Windowing::CWindow;
using Exponent::GUI::Controls::CSubMenuMenuItem;
using Exponent::GUI::Controls::SMenuColours;
using Exponent::GUI::Listeners::IMenuListener;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Windowing
		{
			/**
			 * @class CMenuWindow CMenuWindow.hpp
			 * @brief Specialised window for showing menus
			 *
			 * @date 15/04/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CMenuWindow.hpp,v 1.5 2007/02/08 21:07:54 paul Exp $
			 */
			class CMenuWindow : public CWindow, public IMenuListener
			{
				/** @cond */
				EXPONENT_CLASS_DECLARATION;
				/** @endcond */

//	===========================================================================

			public:

//	===========================================================================

				const static long CMENU_WINDOW_NO_SELECTION = -1;			/**< The window has nothing selected */

//	===========================================================================

				/**
				 * Construction
				 * @param size The size of the window
				 * @param attributes The attributes of the window
				 * @param menu The menu that will be displayed
				 * @param useIcons If true menu will display menu item icons (if available)
				 * @param listener The listener for menu events
				 */
				CMenuWindow(const CRect &size, CWindowAttributes *attributes, CMenu *menu, const bool useIcons = true, IMenuListener *listener = NULL);

				/**
				 * Destruction
				 */
				virtual ~CMenuWindow();

//	===========================================================================

				/**
				 * Open the popup
				 * @param point The *global* window position tht you would like this window to open at
				 */
				virtual void openPopupWindow(const CPoint &point);

				/**
				 * Close the popup
				 */
				virtual void closePopupWindow();

				/**
				 * Window gained the focus
				 * @param window The window that gained the focus
				 */
				virtual void gainedFocus(IWindow *window);

				/**
				 * Window lost the focus
				 * @param window The window that lost focus
				 */
				virtual void lostFocus(IWindow *window);                                 

//	===========================================================================

				/**
				 * Register the menu listener
				 * @param listener The menu event listener
				 */
				virtual void registerMenuListener(IMenuListener *listener) { m_menuListener = listener; }

				/**
				 * Handle a menu opening
				 * @param event The event for the menu that opened
				 * @note Intended for internal use only
				 */
				virtual void handleMenuOpening(const CMenuEvent &event);

				/**
				 * Handle a menu closing
				 * @param event The event for the menu that closed
				 * @note Intended for internal use only
				 */
				virtual void handleMenuClosing(const CMenuEvent &event);

				/**
				 * Handle a menu selection
				 * @param event The menu had something selected
				 * @note Intended for internal use only
				 */
				virtual void handleMenuSelection(const CMenuEvent &event);

//	===========================================================================

				/**
				 * Set the menu
				 * @param menu The menu to be displayed
				 */
				virtual void setMenu(CMenu *menu);
				
				/**
				 * Set the menu and recalculate the size of control root and this window to fit
				 * @param menu The menu to be displayed
				 */ 
				virtual void setMenuAndSize(CMenu *menu);

				/**
				 * Set the font
				 * @note recalculates size
				 * @param font The font to use
				 */
				virtual void setFont(CFont *font);

				/**
				 * Set the window colours
				 * @param colours The colours of the menu items
				 */
				virtual void setMenuColours(const SMenuColours &colours);

				/**
				 * Set the selected entry
				 * @param selection If CMENU_WINDOW_NO_SELECTION Then nothing is selected, else the entry
				 */
				virtual void setCurrentSelection(const long selection = CMENU_WINDOW_NO_SELECTION);

				/**
				 * Use icons
				 * @param icons If true then menu items will have icons shown (if they have them) if false, no icons are shown
				 */
				virtual void useIcons(const bool icons = true);

				/**
				 * Set the minimum Width
				 * @param width The minimum menu width in pixels
				 * @note The window will always be >= width
				 */
				virtual void setMinimumWidth(const long width);

//	===========================================================================

				/**
				 * Set the parentWindow
				 * @param menuWindow The menu window
				 */
				virtual void setParentMenuWindow(CMenuWindow *menuWindow) { m_parentMenuWindow = menuWindow; }

				/**
				 * Prepare for a child window to be lauched
				 * @param launching are we launching a sub menu window
				 */
				void preLauchSubWindow(const bool launching = true) { m_openingSubMenuWindow = launching; }
					
				/**
				 * Launch a child window
				 * @param item The sub menu item that will form the new window
				 * @param control The control that will be used on the sub window
				 */
				void lauchSubWindow(CSubMenuMenuItem *item, IControl *control);

				/**
				 * Kill a sub window
				 */
				void killSubWindow();

//	===========================================================================

			protected:

//	===========================================================================

				/**
				 * @class CMenuRoot CMenuWindow.hpp
				 * @brief Replacement control root that is specific to a menu window.\n
				 * Does lots of handy stuff associated with menus, such as determining its own size
				 */
				class CMenuRoot : public CControlRoot
				{
				public:

					/**
					 * Construction
					 * @param parent The parent menu window, this may be NULL
					 */
					CMenuRoot(IWindow *parent);

					/**
					 * Destruction
					 */
					virtual ~CMenuRoot();

//	===========================================================================

					/**
					 * Draw the root control
					 * @param graphics The graphics handle to draw with
					 */
					virtual void drawRootControl(CGraphics &graphics);

//	===========================================================================

					/**
					 * Handle left button being clicked
					 * @param event The mouse event to handle
					 */
					virtual void handleLeftButtonDown(CMouseEvent &event);

					/**
					 * Handle left button being released
					 * @param event The mouse event to handle
					 */
					virtual void handleLeftButtonUp(CMouseEvent &event);

					/**
					 * Handle the mouse movement
					 * @param event The mouse event to handle
					 */
					virtual void handleMouseMovement(CMouseEvent &event);

					/**
					 * Handle a mouse leaving movement
					 * @param event The mouse event to handle
					 */
					virtual void handleMouseLeavingArea(CMouseEvent &event);

//	===========================================================================

					/**
					 * Set the menu
					 * @param menu The menu to use
					 * @retval CDimension* The total dimension of the mneu
					 */
					virtual CDimension *setMenu(CMenu *menu);

					/**
					 * Get the mneu
					 * @retval CMenu* The menu
					 */
					virtual CMenu *getMenu() const { return m_menu; }

					/**
					 * Set the area
					 * @param area The area of the control root
					 */
					virtual void setRootArea(const CRect &area) { m_rootArea = area; }

//	===========================================================================

					/**
					 * Set the font
					 * @note recalculates size
					 * @param font The font to use
					 */
					virtual void setFont(CFont *font);
					
					/**
					 * Get the font
					 * @retval CFont* The font in use
					 */
					virtual CFont *getFont() { return m_font; }

					/**
					 * Set the window colours
					 * @param colours The colours of the menu items
					 */
					virtual void setMenuColours(const SMenuColours &colours);
					
					/**
					 * Get the menu colours
					 * @retval const SMenuColours& The menu colours
					 */
					virtual const SMenuColours &getMenuColours() const { return m_colours; }

//	===========================================================================

					/**
					 * Set the selected entry
					 * @param selection If CMENU_WINDOW_NO_SELECTION Then nothing is selected, else the entry
					 */
					virtual void setCurrentSelection(const long selection = CMENU_WINDOW_NO_SELECTION);

//	===========================================================================

					/**
					 * Use icons
					 * @param icons If true then menu items will have icons shown (if they have them) if false, no icons are shown
					 */
					virtual void useIcons(const bool icons = true);

					/**
					 * Set the minimum Width
					 * @param width The minimum menu width in pixels
					 * @note The window will always be >= width
					 */
					virtual void setMinimumWidth(const long width);

//	===========================================================================

				protected:

//	===========================================================================

					/**
					 * Find the biggest dimensions in the string sthat we have
					 * @param maxDimension On return stores the max dimension string we are using
					 */
					void findMaxStringDimension(CDimension &maxDimension);

					/**
					 * Find the max icon dimensions
					 * @param maxDimension On return holds the max dimensions of the images contained within the menu
					 * @retval bool True if icons were found, false otherwise
					 */
					bool findMaxIconDimension(CDimension &maxDimension);

					/**
					 * Does it have a sub menu item
					 * @param selectable On return is true if selectable items found, false otherwise
					 * @retval bool True if sub menus found, false ohterwise
					 */
					bool doesMenuHaveExtraItems(bool &selectable);



//	===========================================================================
					
					CMenu *m_menu;								/**< The menu */
					CFont *m_font;								/**< The font to use */

					//long m_itemHeight;							/**< Height of a single item */
					//long m_itemWidth;							/**< Width of an item */
					long m_minimumWidth;						/**< Width should be >= this */
					long m_currentSelection;					/**< The currently selected item */
					long m_guardSelection;						/**< Item selected on mouse down */

					SMenuColours m_colours;						/**< The menu colours */

					CRect m_rootArea;							/**< The are of the root control */
					bool m_useIcons;							/**< Should icons be used? */

				};

//	===========================================================================

				IMenuListener *m_menuListener;					/**< Listener for menu actions */
				CDimension *m_mySizeOnInitialise;				/**< The size of the menu window on initialise */
				CMenuWindow *m_subWindow;						/**< The sub window */
				CMenuWindow *m_parentMenuWindow;				/**< The parent menu */
				bool m_openingSubMenuWindow;					/**< Are we currently opening a sub window */
				bool m_useIcons;								/**< Should icons be used? */
				/** @cond */
				#ifndef WIN32
				/** @endcond */
					bool m_closingWindow;						/**< Are we closing the window */
					WindowGroupRef m_windowGroup;				/**< Menu window group */
				/** @cond */
				#endif
				/** @endcond */

			};
		}
	}
}
#endif	// End of CMenuWindow.hpp